package org.amse.mm.myVirtualBilliards.model.impl;

import java.util.*;
import org.amse.mm.io.*;
import org.amse.mm.myVirtualBilliards.model.*;

public class Table implements ITable{
	private LinkedList<IBall> myBallList;
	private double numericalError = 0.000000001;
	private double myFriction;
	
	public Table(){
		myBallList = new LinkedList<IBall>();
		ReadFromJar reader = new ReadFromJar(this, "initialize/friction.bll", false);
		reader.init();
	}
	
	public Table(double friction){
		myBallList = new LinkedList<IBall>();
		myFriction = friction;
	}
	
	public void deleteBall(IBall ball){
		myBallList.remove(ball);
	}
	
	public void setFriction(double friction){
		myFriction = friction;
	}
	
	public double getFriction(){
		return myFriction;
	}
	
	public void clearTable(){
		myBallList.clear();
	}
	
	public IBall addBall(double x, double y, BallColor color){
		if (canAddBall(x, y, color)){
			Ball ball = new Ball(x, y, color);
			myBallList.add(ball);
			return ball;
		}
		return null;
	}
	
	private boolean canAddBall(double x, double y, BallColor color){
		if ((x < 0) || (x > WIDTH_TABLE) || (y < 0) || (y > LENGTH_TABLE)){
			return false;
		}
		for(IBall ball : myBallList){
			if ((Math.hypot(x - ball.getCoordinate().X, 
					   	    y - ball.getCoordinate().Y) < DIAMETER_BALL) ||
						    (color == ball.getColor())) {
				return false;
			}
		} 
		return true;
	}
	
	public List<IBall> balls(){
		return Collections.unmodifiableList(myBallList);
	}
	
	private void moveBall(IBall ball, double time){
		ball.setCoordinate(ball.getCoordinate().X + (1 / myFriction) *
				   ball.getVelocity().VX * (1 - Math.exp(-myFriction * time)), 
				   ball.getCoordinate().Y + (1 / myFriction) * 
				   ball.getVelocity().VY * (1 - Math.exp(-myFriction * time)));
		ball.setVelocity(ball.getVelocity().VX * Math.exp(-myFriction * time),
				 ball.getVelocity().VY * Math.exp(-myFriction * time));
		
		if (Math.hypot(ball.getVelocity().VX, ball.getVelocity().VY) < 3){
			ball.setVelocity(0, 0);
		}
	}
	
	public void doMove(double interval){
		double time = timeBeforeStrike();
		while(interval >= time){			
			for (IBall ball : myBallList){
				if ((ball.getVelocity().VX != 0) || (ball.getVelocity().VY != 0)){
					moveBall(ball, time);
				}				
			}		
			interval = interval - time;
			downtrodden();
			strikeBall();
			time = timeBeforeStrike();
		}
		
		for (IBall ball : myBallList){
			if ((ball.getVelocity().VX != 0) || (ball.getVelocity().VY != 0)){
				moveBall(ball, interval);
			}				
		}		
	}
	
	
	private double findRadicalValue(double a, double b, double c){
		double time = Double.MAX_VALUE;
		double d = (b * b) - (4 * a * c);
		if ((-b - Math.sqrt(d)) / (2 * a) > 0){
			time = (-b - Math.sqrt(d)) / (2 * a);
		}
		return time;
	}
	
	private double timeBeforeStrikeOfBall(IBall ball){
		double time = Double.MAX_VALUE;
		for (IBall tmpBall : myBallList){
			double tmpTime;
			if (ball != tmpBall){												
				double tmp = findRadicalValue(
						   (Math.pow((ball.getVelocity().VX - tmpBall.getVelocity().VX) / myFriction, 2) +
							Math.pow((ball.getVelocity().VY - tmpBall.getVelocity().VY) / myFriction, 2)),
							2 * ((ball.getCoordinate().X - tmpBall.getCoordinate().X) *
								 (ball.getVelocity().VX - tmpBall.getVelocity().VX) +
								 (ball.getCoordinate().Y - tmpBall.getCoordinate().Y) *
								 (ball.getVelocity().VY - tmpBall.getVelocity().VY)) / myFriction,
							(Math.pow(ball.getCoordinate().X - tmpBall.getCoordinate().X, 2) +
							 Math.pow(ball.getCoordinate().Y - tmpBall.getCoordinate().Y, 2) - 
							 Math.pow(DIAMETER_BALL, 2)));
				tmpTime = - (1 / myFriction) * Math.log(1 - tmp);
		
				if(tmpTime < time){
					time = tmpTime;
				}
			}
		}
		return time;
	}	
	
	private double timeBeforeStrikeOfBoard(IBall ball){
		double tX = Double.MAX_VALUE;
		double tY = Double.MAX_VALUE;
		double tmp;
		if (ball.getVelocity().VX > 0){
			tmp = - (1 / myFriction) * Math.log(1 - myFriction * (WIDTH_TABLE - ball.getCoordinate().X) / ball.getVelocity().VX);
			if (tmp < tX){
				tX = tmp;
			}
		}else{
			if (ball.getVelocity().VX < 0){
				tmp = - (1 / myFriction) * Math.log(1 + myFriction * (ball.getCoordinate().X) / ball.getVelocity().VX);
				if (tmp < tX){
					tX = tmp;
				}
			}
		}
		
		if (ball.getVelocity().VY > 0){
			tmp = - (1 / myFriction) * Math.log(1 - myFriction * (LENGTH_TABLE - ball.getCoordinate().Y) / ball.getVelocity().VY);
			if (tmp < tY){
				tY = tmp;
			}
		}else{
			if (ball.getVelocity().VY < 0){
				tmp = - (1 / myFriction) * Math.log(1 + myFriction * (ball.getCoordinate().Y) / ball.getVelocity().VY);
				if (tmp < tY){
					tY = tmp;
				}
			}
		}
		return Math.min(tX, tY);
	}
	
	private double timeBeforeStrike(){
		double timeBall = Double.MAX_VALUE;
		double timeBoard = Double.MAX_VALUE;
		for(IBall ball : myBallList){
			double tmpTimeBall = timeBeforeStrikeOfBoard(ball);
			double tmpTimeBoard = timeBeforeStrikeOfBall(ball);
			if (timeBall > tmpTimeBall){
				timeBall = tmpTimeBall;
			}
			if (timeBoard > tmpTimeBoard){
				timeBoard = tmpTimeBoard;
			}
		}
		
		return Math.min(timeBoard, timeBall);
	}
	
	private void strikeWithBoard(IBall ball){
		if (((Math.abs(ball.getCoordinate().X) < numericalError) && (ball.getVelocity().VX < 0))|| 
			((Math.abs(ball.getCoordinate().X - WIDTH_TABLE) < numericalError) && (ball.getVelocity().VX > 0))){
			ball.setVelocity(-ball.getVelocity().VX, ball.getVelocity().VY);
		}
		if (((Math.abs(ball.getCoordinate().Y) <= numericalError) && (ball.getVelocity().VY < 0)) || 
			((Math.abs(ball.getCoordinate().Y - LENGTH_TABLE) < numericalError) && (ball.getVelocity().VY > 0))){
			ball.setVelocity(ball.getVelocity().VX, -ball.getVelocity().VY);
		}		
	}
	
	private double findAngle(double Vx, double Vy){
		double angle = Math.atan(Vy / Vx);
		if ((Vx < 0) && (Vy < 0)){
			angle = angle - Math.PI;  
		}else{
			if ((Vx < 0) && (Vy > 0)){
				angle = angle + Math.PI;
			}
		}
		return angle;
	}
		
	public void strikeWithBall(IBall ball1, IBall ball2){
		double tmpVx = ball2.getVelocity().VX;
		double tmpVy = ball2.getVelocity().VY;
		ball1.getVelocity().VX = ball1.getVelocity().VX - tmpVx;
		ball1.getVelocity().VX = ball1.getVelocity().VX - tmpVy;
		double angleX = Math.PI - 2 * Math.asin(findPinpointParameter(ball1, ball2) / DIAMETER_BALL);
		double moduleV = Math.hypot(ball1.getVelocity().VX, ball1.getVelocity().VY);
		double moduleV1Finish = moduleV * Math.cos(angleX / 2);
		double moduleV2Finish = moduleV * Math.sin(angleX / 2);
		
		double angle = findAngle(ball1.getVelocity().VX, ball1.getVelocity().VY);
		double angleThetta1 = angleX / 2;
		double angleThetta2 = (Math.PI - angleX) / 2;
	
		double sign; 
		double tmpAngle = findAngle(ball2.getCoordinate().X - ball1.getCoordinate().X,
						   ball2.getCoordinate().Y - ball1.getCoordinate().Y);
		
		if (((angle >= Math.PI / 4) && (angle <= 3 * Math.PI / 4)) ||
			((angle <= -Math.PI / 4) && (angle >= -3 * Math.PI / 4))){
			sign = Math.signum(ball2.getCoordinate().X - ball1.getCoordinate().X) *
				   Math.signum(angle);
		}else{
			sign = Math.signum(ball1.getCoordinate().Y - ball2.getCoordinate().Y) *
			   	   Math.signum(Math.cos(angle)) ;
		}
		
		
		
		if (((angle > 0) && (angle < Math.PI/4)) && ((tmpAngle > 0) && (tmpAngle < Math.PI/4)) ||
			((angle > Math.PI/2) && (angle < 3*Math.PI/4)) && ((tmpAngle > Math.PI/2) && (tmpAngle < 3*Math.PI/4)) ||
			((angle > -Math.PI/2) && (angle < -Math.PI/4)) && ((tmpAngle > -Math.PI/2) && (tmpAngle < -Math.PI/4)) ||
			((angle > -Math.PI) && (angle < -3*Math.PI/4)) && ((tmpAngle > -Math.PI) && (tmpAngle < -3*Math.PI/4))){
			
			if (tmpAngle < angle){
				sign = -sign;
			}
		}else if (((angle > Math.PI/4) && (angle < Math.PI/2)) && ((tmpAngle > Math.PI/4) && (tmpAngle < Math.PI/2)) ||
				 ((angle > 3*Math.PI/4) && (angle < Math.PI)) && ((tmpAngle > 3*Math.PI/4) && (tmpAngle < Math.PI)) ||
				 ((angle > -Math.PI/4) && (angle < 0)) && ((tmpAngle > -Math.PI/4) && (tmpAngle < 0)) || 
				 ((angle > -3*Math.PI/4) && (angle < -Math.PI/2)) && ((tmpAngle > -3*Math.PI/4) && (tmpAngle < -Math.PI/2))){
			
			if (tmpAngle > angle){
				sign = -sign;
			}
		}
			
		
		ball1.setVelocity(moduleV1Finish * Math.cos(angle + sign * angleThetta1) + tmpVx,
						  moduleV1Finish * Math.sin(angle + sign * angleThetta1) + tmpVy);
		ball2.setVelocity(moduleV2Finish * Math.cos(angle - sign * angleThetta2) + tmpVx,
						  moduleV2Finish * Math.sin(angle - sign * angleThetta2) + tmpVy);
	}
	
	private double findPinpointParameter(IBall ball1, IBall ball2){
		double tmpTime;
		tmpTime = ((ball1.getVelocity().VX * (ball2.getCoordinate().X - ball1.getCoordinate().X) +
				    ball1.getVelocity().VY * (ball2.getCoordinate().Y - ball1.getCoordinate().Y)) /
				    Math.pow(Math.hypot(ball1.getVelocity().VX, ball1.getVelocity().VY), 2));
		double pinpoint =  Math.hypot(ball2.getCoordinate().X - ball1.getCoordinate().X - ball1.getVelocity().VX * tmpTime,
						  ball2.getCoordinate().Y - ball1.getCoordinate().Y - ball1.getVelocity().VY * tmpTime);
		return pinpoint;
	}
	
	private void strikeBall(){
		for (IBall ball : myBallList){
			strikeWithBoard(ball);
		}	
		
		
		for (IBall ball1 : myBallList){
			for (IBall ball2 : myBallList){
				if (ball2 == ball1) break;
				if (Math.abs(Math.hypot(ball1.getCoordinate().X - ball2.getCoordinate().X, 
				    ball1.getCoordinate().Y - ball2.getCoordinate().Y) - DIAMETER_BALL) < numericalError){
					if (Math.hypot(ball2.getVelocity().VX, ball2.getVelocity().VY) >=
						Math.hypot(ball1.getVelocity().VX, ball1.getVelocity().VY))	{	
						strikeWithBall(ball2, ball1);
					}else{
						strikeWithBall(ball1, ball2);
					}
				}
			}
		}
	}
	
	public void downtrodden(){
		for (ListIterator it = myBallList.listIterator(); it.hasNext();){
			IBall ball = (IBall)it.next();
			if (((Math.hypot(ball.getCoordinate().X, ball.getCoordinate().Y - (LENGTH_TABLE / 2)) < 4) && (ball.getVelocity().VX < 0)) ||
				((Math.hypot(ball.getCoordinate().X - WIDTH_TABLE, ball.getCoordinate().Y - (LENGTH_TABLE / 2)) < 4) && (ball.getVelocity().VX > 0)) ||
				((Math.hypot(ball.getCoordinate().X, ball.getCoordinate().Y) < 4) && (ball.getVelocity().VX < 0) && (ball.getVelocity().VY < 0)) ||
				((Math.hypot(ball.getCoordinate().X, ball.getCoordinate().Y - LENGTH_TABLE) < 4) && (ball.getVelocity().VX < 0) && (ball.getVelocity().VY > 0)) ||
				((Math.hypot(ball.getCoordinate().X - WIDTH_TABLE, ball.getCoordinate().Y) < 4) && (ball.getVelocity().VX > 0) && (ball.getVelocity().VY < 0)) ||
				((Math.hypot(ball.getCoordinate().X - WIDTH_TABLE, ball.getCoordinate().Y - LENGTH_TABLE) < 4) && (ball.getVelocity().VX > 0) && (ball.getVelocity().VY > 0))){
				it.remove();
			}
		}
	}
	
	public boolean areBallsMoving(){
		for(IBall ball : myBallList){
			if ((ball.getVelocity().VX != 0) || (ball.getVelocity().VY != 0)){
				return true;
			}
		}
		return false;		
	}

}
